/**
 * @file selenium.keys.ts - a proprietary library created for the 
 * implementation of the Medical Care Collection Fund 
 * Electronic Data Interchange Transaction Application Suite 
 * (MCCF EDI TAS) for easily automating web applications for 
 * testing purposes.  Part of a larger keyword-driven test 
 * automation framework, it's designed with the objective of 
 * successfully porting Robot Framework's (RF) "Selenium2Library".  
 * 
 * Adaption to Angular 2 and TypeScript from the original, 
 * written in Python, designed for Python, and Jython 
 * interpreters, introduces various nuances.  However, utilization 
 * of the particulars concerning the keyword-driven test scripting 
 * techniques in this library empowers testers coming from  
 * non-programming backgrounds to automate and manage complex 
 * workflow scripts efficiently.
 *  
 * The function(s) within this file constitute keywords describing 
 * specific sets of actions required to perform individual, and 
 * agnostic, test instructions (i.e., not directly linked to the 
 * application within MCCF EDI TAS, and flexible enough to work in 
 * most modern browsers).
 * 
 * author David Azzolina
 * license MIT
 * copyright U.S. Department of Veterans Affairs
 *   
 * @summary A Selenium 2 (WebDriver)/Protractor library wrapper.
 * @requires mocha.js, chai.js, protractor.js, built-in.keys.ts  
 * @link http://robotframework.org/Selenium2Library/Selenium2Library.html
 */

import * as ptor from 'protractor';

import { jasmineSpecHelper } from './jasmine-spec-helper.keys';
import { BuiltIn } from './built-in.keys';
 /**
 * @class Selenium2Library
 * @classdesc Web testing library for MCCF EDI TAS.
 */
export class Selenium2Library {
	/** 
 	 * @constructs Selenium2Library 
 	 * @param private logger Session's instance of a winston logger.
 	 * @param private comparison Instantiates BuiltIn library for verifications.
 	 * @param private number defaultTimeout Globally scoped timeout for any keywords with timeout arguments.
 	 */
	constructor(
		  private logger = null
		, private comparison = new BuiltIn()
		, private defaultTimeout: number = 5000) 
	{}
	/**
	 * @summary Input Text.
	 *
	 * Types the given text into text field identified by locator.
	 *
	 * @link http://robotframework.org/Selenium2Library/Selenium2Library.html#Input%20Text
	 * @param string | ptor.ElementFinder css Locator.
	 * @param string text 
	 */
	inputText(css: string | ptor.ElementFinder, text: string) {
		/* 
	 	 * Preserve original "this" object that the method was called on 
	 	 * in "that". Otherwise, "this" is undefined inside a class method 
	 	 * when using promises.
	 	 */  
		let that: Selenium2Library = this;
		let elem;
		if (css && typeof css === "string") 
		{
			/* Find the element using Protractor's shortcut $() notation.*/	  
			elem = ptor.$(css);
		}
		else 
		{
			elem = css; 
		}		
		/* Waits "defaultTimeout" for the element to appear.*/
		ptor.browser.wait(
			ptor.ExpectedConditions.presenceOf(elem), this.defaultTimeout) //TO-DO: pageShouldContainElement() 
			.then(function () {  		
				/*Logs metadata for about this test step.*/ 
				that.getLoggerInstance().log(
					'info'
					, 'Selenium2Library.inputText' + css
					, 'keyword');
				that.getLoggerInstance().log(
					'info'
					, 'Sending Keys \'' + text + ' to ' + css + '\'.'
					, 'keyword');
				/*Sends keys into the input field.*/
				elem.sendKeys(text);
			}
		);
	}
		/**
	 * @summary Input Text.
	 *
	 * Types the given text into text field identified by locator.
	 *
	 * @link http://robotframework.org/Selenium2Library/Selenium2Library.html#Input%20Text
	 * @param string | ptor.ElementFinder css Locator.
	 * @param string text 
	 */
	goTo(url = '') {
		var that = this;  //this is always the object the method is called on; therefore, "this" is undefined inside a class method when using promises.  
		// If you want to preserve this, you will have to do it like this:
			//.then(() => this.method2())
			//Or the pre-ES6 way I did above.
		ptor.browser.getProcessedConfig().then(function (conf) {
  				let browserName = conf.capabilities.browserName;
  				let baseURL = that.getBrowserBaseURL();
				let addr = (baseURL + url);

				that.getLoggerInstance().log('info', 'Selenium2Library.goTo ' + addr + ', ' + browserName, 'keyword');
				ptor.browser.get(url);
				that.getLoggerInstance().log('info', 'Opening browser \'' + browserName + '\' to base url \'' + addr + '\'', 'keyword');
				
				ptor.browser.driver.getCurrentUrl().then(function(currURL) {
		      		that.locationShouldBe(currURL);
		      		expect(addr).toEqual(currURL);//Tests that the URL has been reached and is now the current URL.
		    		that.getLoggerInstance().log('info', 'CurrentURL verified: ' + currURL, 'result');
		    	});
		    });
	}
	/**
* TO-DO
*/
	setLoggerInstance(logger) {
		this.logger = logger;
		this.comparison.setLoggerInstance(logger);
	}
	getLoggerInstance() {
		return this.logger;
	}
	getBrowserBaseURL() {
		return ptor.browser.baseUrl;
	}
	locationShouldBe(url) {
 		ptor.browser.wait(ptor.ExpectedConditions.urlContains(url), 5000);
 		// Do more by using should be equal
 	}
}